// Content script for grade input extension

let inputFields = [];
let currentIndex = 0;

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'ping') {
    sendResponse({ pong: true });
    return false;
  } else if (request.action === 'fillGrades') {
    fillGrades(request.grades, sendResponse);
    return true; // Will respond asynchronously
  } else if (request.action === 'checkPage') {
    checkPage(sendResponse);
    return false;
  } else if (request.action === 'clearGrades') {
    clearGrades(sendResponse);
    return true; // Will respond asynchronously
  } else if (request.action === 'getPageData') {
    getPageData(sendResponse);
    return true; // Will respond asynchronously
  }
});

function findInputFields() {
  inputFields = [];

  // Look for various types of grade input fields
  const possibleSelectors = [
    // Direct input fields
    'input[type="text"][name*="Grade"]',
    'input[type="text"][name*="grade"]',
    'input[type="number"][name*="Grade"]',
    'input[type="number"][name*="grade"]',
    'input[type="text"][class*="grade"]',
    'input[type="number"][class*="grade"]',

    // Struts2 form fields (common in academic systems)
    'input[name*="examGradeState"]',
    'input[name*="GradeState"]',

    // Editable table cells
    'td[contenteditable="true"]',
    'div[contenteditable="true"]',

    // Common grade input patterns
    'input.txtScore',
    'input.score-input',
    'input.grade-input',

    // All text/number inputs in forms
    'form input[type="text"]',
    'form input[type="number"]'
  ];

  // Try each selector
  for (const selector of possibleSelectors) {
    const fields = document.querySelectorAll(selector);
    fields.forEach(field => {
      // Check if it's visible and enabled
      if (isVisible(field) && !field.disabled && !field.readOnly) {
        // Avoid duplicates
        const existing = inputFields.find(f => f.name === field.name && f.value === field.value);
        if (!existing) {
          inputFields.push({
            element: field,
            name: field.name || field.className,
            value: field.value
          });
        }
      }
    });
  }

  // Also try to find by table structure
  const tables = document.querySelectorAll('table');
  tables.forEach(table => {
    const rows = table.querySelectorAll('tr');
    rows.forEach(row => {
      const cells = row.querySelectorAll('td');
      cells.forEach((cell, index) => {
        // Check if cell looks like a grade cell
        const input = cell.querySelector('input[type="text"], input[type="number"]');
        if (input && isVisible(input)) {
          // Only add if not already found
          const exists = inputFields.find(f => f.element === input);
          if (!exists && !input.disabled && !input.readOnly) {
            inputFields.push({
              element: input,
              name: input.name || `cell_${index}`,
              value: input.value,
              parentCell: cell
            });
          }
        }
      });
    });
  });

  return inputFields;
}

function isVisible(element) {
  const style = window.getComputedStyle(element);
  return style.display !== 'none' &&
         style.visibility !== 'hidden' &&
         style.opacity !== '0' &&
         element.offsetParent !== null;
}

function fillGrades(grades, sendResponse) {
  try {
    // Find all input fields
    const fields = findInputFields();

    if (fields.length === 0) {
      sendResponse({
        success: false,
        error: '未找到成绩输入框，请确保页面已加载完成'
      });
      return;
    }

    let filledCount = 0;
    currentIndex = 0;

    // Fill grades sequentially with delay
    function fillNext() {
      if (currentIndex < grades.length && currentIndex < fields.length) {
        const grade = grades[currentIndex];
        const field = fields[currentIndex];

        // Fill the input
        field.element.value = grade.totalScore;

        // Trigger change events
        const event = new Event('change', { bubbles: true });
        field.element.dispatchEvent(event);

        const inputEvent = new Event('input', { bubbles: true });
        field.element.dispatchEvent(inputEvent);

        filledCount++;
        currentIndex++;

        // Delay before next fill
        setTimeout(fillNext, 100);
      } else {
        sendResponse({
          success: true,
          count: filledCount,
          totalGrades: grades.length,
          totalFields: fields.length
        });
      }
    }

    fillNext();

  } catch (error) {
    sendResponse({
      success: false,
      error: '录入失败: ' + error.message
    });
  }
}

function checkPage(sendResponse) {
  const fields = findInputFields();
  sendResponse({
    hasInputFields: fields.length > 0,
    inputCount: fields.length,
    fieldNames: fields.slice(0, 10).map(f => f.name) // Send first 10 names
  });
}

function clearGrades(sendResponse) {
  try {
    // Find all input fields
    const fields = findInputFields();

    if (fields.length === 0) {
      sendResponse({
        success: false,
        error: '未找到成绩输入框'
      });
      return;
    }

    let clearedCount = 0;
    currentIndex = 0;

    // Clear grades sequentially with delay
    function clearNext() {
      if (currentIndex < fields.length) {
        const field = fields[currentIndex];

        // Only clear if field has a value
        if (field.element.value && field.element.value !== '') {
          field.element.value = '';

          // Trigger change events
          const event = new Event('change', { bubbles: true });
          field.element.dispatchEvent(event);

          const inputEvent = new Event('input', { bubbles: true });
          field.element.dispatchEvent(inputEvent);

          clearedCount++;
        }

        currentIndex++;

        // Delay before next clear
        setTimeout(clearNext, 50);
      } else {
        sendResponse({
          success: true,
          count: clearedCount
        });
      }
    }

    clearNext();

  } catch (error) {
    sendResponse({
      success: false,
      error: '清除失败: ' + error.message
    });
  }
}

function getPageData(sendResponse) {
  try {
    const students = [];

    // Try to find student data using various patterns
    // Pattern 1: Struts2 indexed properties like examGradeState[0].student.code
    const indexedPattern = /examGradeState\[(\d+)\]\.(\w+)/;
    const allInputs = document.querySelectorAll('input[name]');

    // Collect all indexed fields
    const indexedFields = {};
    allInputs.forEach(input => {
      const match = input.name.match(indexedPattern);
      if (match) {
        const index = parseInt(match[1]);
        const fieldType = match[2]; // e.g., 'student.code', 'examGrade.percent'
        if (!indexedFields[index]) {
          indexedFields[index] = {};
        }
        indexedFields[index][fieldType] = input.value;
      }
    });

    // Convert to student array
    Object.keys(indexedFields).sort((a, b) => parseInt(a) - parseInt(b)).forEach(index => {
      const fields = indexedFields[index];
      const studentNum = fields['student.code'] || fields['student.std.code'] || fields['studentNo'] || '';
      const studentName = fields['student.name'] || fields['student.std.name'] || fields['studentName'] || '';
      const totalScore = fields['examGrade.percent'] || fields['score'] || fields['grade'] || '';

      students.push({
        index: parseInt(index) + 1,
        studentNum: studentNum,
        studentName: studentName,
        totalScore: totalScore
      });
    });

    // Pattern 2: Table-based student data
    if (students.length === 0) {
      const tables = document.querySelectorAll('table');
      tables.forEach(table => {
        const rows = table.querySelectorAll('tr');
        rows.forEach((row, rowIndex) => {
          const cells = row.querySelectorAll('td, th');
          if (cells.length >= 3) {
            // Try to extract student info from table
            const studentNumCell = cells[0].textContent.trim();
            const studentNameCell = cells[1].textContent.trim();
            const scoreCell = cells[cells.length - 1].textContent.trim();

            // Check if first cell looks like a student number (numeric)
            if (/^\d+$/.test(studentNumCell) && studentNumCell.length > 0) {
              students.push({
                index: rowIndex + 1,
                studentNum: studentNumCell,
                studentName: studentNameCell,
                totalScore: scoreCell
              });
            }
          }
        });
      });
    }

    // Pattern 3: Find by table row structure with input fields
    if (students.length === 0) {
      const tables = document.querySelectorAll('table');
      tables.forEach(table => {
        const rows = table.querySelectorAll('tr');
        rows.forEach((row, rowIndex) => {
          const inputs = row.querySelectorAll('input[type="text"], input[type="number"]');
          if (inputs.length > 0) {
            // This row might have grade inputs
            const firstInput = inputs[0];
            const name = firstInput.name || '';

            // Try to extract index from name
            const indexMatch = name.match(/\[(\d+)\]/);
            if (indexMatch) {
              const index = parseInt(indexMatch[1]);
              const scoreInput = inputs[inputs.length - 1]; // Last input is likely the score

              students.push({
                index: index + 1,
                studentNum: '',
                studentName: '',
                totalScore: scoreInput ? scoreInput.value : ''
              });
            }
          }
        });
      });
    }

    if (students.length === 0) {
      // Fallback: find all grade input fields and use their values
      const fields = findInputFields();
      fields.forEach((field, idx) => {
        students.push({
          index: idx + 1,
          studentNum: '',
          studentName: '',
          totalScore: field.value
        });
      });
    }

    sendResponse({
      success: true,
      students: students,
      count: students.length
    });

  } catch (error) {
    sendResponse({
      success: false,
      error: '获取页面数据失败: ' + error.message
    });
  }
}

// Also try to find fields on page load
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    setTimeout(() => findInputFields(), 500);
  });
} else {
  setTimeout(() => findInputFields(), 500);
}
