// Global state
let gradesData = null;

// ==================== 授权验证模块 ====================
// 密钥（用于简单混淆，防止直接修改）
const AUTH_SECRET = 'GradeInput2024';

// API地址
const AUTH_API_URL = 'https://nongbin.site/api/verify-auth';

// 授权码格式：Base64(JSON({expireDate: timestamp, salt: "xxx"}))
function generateAuthCode(expireDays = 14) {
  const expireDate = Date.now() + expireDays * 24 * 60 * 60 * 1000;
  const salt = Math.random().toString(36).substring(2, 8);
  const data = JSON.stringify({ expireDate, salt });
  return btoa(encrypt(data, AUTH_SECRET));
}

// 简单异或加密
function encrypt(str, key) {
  let result = '';
  for (let i = 0; i < str.length; i++) {
    result += String.fromCharCode(str.charCodeAt(i) ^ key.charCodeAt(i % key.length));
  }
  return result;
}

// 解密
function decrypt(str, key) {
  try {
    let result = '';
    for (let i = 0; i < str.length; i++) {
      result += String.fromCharCode(str.charCodeAt(i) ^ key.charCodeAt(i % key.length));
    }
    return result;
  } catch (e) {
    return null;
  }
}

// 本地验证授权码
function verifyAuthCodeLocal(code) {
  try {
    const decoded = atob(code);
    const jsonStr = decrypt(decoded, AUTH_SECRET);
    if (!jsonStr) return { valid: false, error: '授权码无效' };

    const data = JSON.parse(jsonStr);
    if (data.expireDate && Date.now() > data.expireDate) {
      return { valid: false, error: '授权已过期', expired: true };
    }
    return { valid: true, expireDate: data.expireDate };
  } catch (e) {
    return { valid: false, error: '授权码格式错误' };
  }
}

// 在线验证授权码
async function verifyAuthCodeOnline(code) {
  try {
    const response = await fetch(AUTH_API_URL, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ code: code })
    });

    const result = await response.json();

    if (result.valid) {
      return {
        valid: true,
        expireDate: result.expireDate,
        softwareId: result.softwareId,
        onlineVerified: true
      };
    } else {
      return {
        valid: false,
        error: result.error || '在线验证失败',
        expired: result.expired || false,
        onlineVerified: false
      };
    }
  } catch (error) {
    // 网络错误时降级使用本地验证
    console.warn('在线验证失败，降级到本地验证:', error);
    return { valid: null, error: '网络异常，使用本地验证', offline: true };
  }
}

// 检查授权状态（先本地验证，再在线验证，必须全部通过）
async function checkAuth() {
  const result = await new Promise((resolve) => {
    chrome.storage.local.get(['authCode'], (result) => {
      resolve(result.authCode);
    });
  });

  const code = result;
  if (!code) {
    return { authorized: false, firstUse: true };
  }

  // 先进行本地验证
  const localResult = verifyAuthCodeLocal(code);
  if (!localResult.valid) {
    if (localResult.expired) {
      return { authorized: false, expired: true, error: '授权已过期' };
    }
    return { authorized: false, invalid: true, error: localResult.error };
  }

  // 本地验证通过，进行在线验证
  const onlineResult = await verifyAuthCodeOnline(code);

  if (onlineResult.valid) {
    // 在线验证成功
    return {
      authorized: true,
      expireDate: onlineResult.expireDate,
      onlineVerified: true
    };
  } else if (onlineResult.offline) {
    // 网络异常，在线验证失败，拒绝使用
    return {
      authorized: false,
      error: '网络异常，请检查网络连接后重试',
      offline: true
    };
  } else {
    // 在线验证失败（授权码无效或过期）
    return {
      authorized: false,
      error: onlineResult.error,
      expired: onlineResult.expired
    };
  }
}

// 保存授权码
function saveAuthCode(code) {
  return new Promise((resolve) => {
    chrome.storage.local.set({ authCode: code }, resolve);
  });
}

// 授权验证UI控制
function showAuthSection(firstUse = false, error = null) {
  document.getElementById('authSection').classList.add('active');
  document.getElementById('mainContent').classList.add('disabled');

  const expiredMsg = document.getElementById('authExpiredMsg');
  const firstUseMsg = document.getElementById('authFirstUseMsg');
  const authStatus = document.getElementById('authStatus');

  if (firstUse) {
    expiredMsg.style.display = 'none';
    firstUseMsg.style.display = 'block';
    authStatus.style.display = 'none';
  } else {
    // 非首次使用，显示错误信息
    expiredMsg.style.display = 'none';
    firstUseMsg.style.display = 'none';
    if (error) {
      authStatus.textContent = error;
      authStatus.className = 'status error';
      authStatus.style.display = 'block';
    } else {
      authStatus.style.display = 'none';
    }
  }
}

function showMainSection() {
  document.getElementById('authSection').classList.remove('active');
  document.getElementById('mainContent').classList.remove('disabled');
}

// 授权按钮事件（放在外部，确保总是绑定）
document.addEventListener('DOMContentLoaded', function() {
  const authBtn = document.getElementById('authBtn');
  const authCodeInput = document.getElementById('authCode');
  const authStatus = document.getElementById('authStatus');

  if (authBtn) {
    authBtn.addEventListener('click', async () => {
      const code = authCodeInput.value.trim();
      if (!code) {
        authStatus.textContent = '请输入授权码';
        authStatus.className = 'status error';
        authStatus.style.display = 'block';
        return;
      }

      // 显示验证中状态
      authBtn.disabled = true;
      authStatus.textContent = '正在验证授权码...';
      authStatus.className = 'status info';
      authStatus.style.display = 'block';

      // 先进行本地验证
      const localResult = verifyAuthCodeLocal(code);
      if (!localResult.valid) {
        authStatus.textContent = localResult.error;
        authStatus.className = 'status error';
        authBtn.disabled = false;
        return;
      }

      // 本地验证通过，进行在线验证
      const onlineResult = await verifyAuthCodeOnline(code);

      if (onlineResult.valid) {
        // 在线验证成功
        await saveAuthCode(code);
        showMainSection();
        showAuthVerifiedStatus({ onlineVerified: true });
        initMainFeatures();
      } else if (onlineResult.offline) {
        // 网络异常，在线验证失败，拒绝使用
        authStatus.textContent = '网络异常，请检查网络连接后重试';
        authStatus.className = 'status error';
        authBtn.disabled = false;
      } else {
        // 在线验证失败（授权码无效或过期）
        authStatus.textContent = onlineResult.error;
        authStatus.className = 'status error';
        authBtn.disabled = false;
      }
    });
  }

  // 按回车也能验证
  if (authCodeInput) {
    authCodeInput.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') {
        authBtn.click();
      }
    });
  }
});

document.addEventListener('DOMContentLoaded', async function() {
  // 先检查授权
  const authStatus = await checkAuth();

  if (authStatus.authorized) {
    showMainSection();
    showAuthVerifiedStatus(authStatus);
    initMainFeatures();  // 授权通过，初始化主功能
  } else {
    // 显示错误信息（首次使用除外）
    if (!authStatus.firstUse) {
      showAuthSection(false, authStatus.error || '授权验证失败');
    } else {
      showAuthSection(true);
    }
    // 不执行 initMainFeatures，授权区域会显示
  }
});

// 显示授权验证状态
function showAuthVerifiedStatus(authStatus) {
  const statusEl = document.getElementById('authVerifiedStatus');
  if (!statusEl) return;

  if (authStatus.onlineVerified) {
    statusEl.style.display = 'block';
    statusEl.style.background = '#e8f5e9';
    statusEl.style.color = '#2e7d32';
    statusEl.textContent = '授权验证通过';
  } else {
    statusEl.style.display = 'none';
  }
}

// 初始化主功能（授权通过后调用）
function initMainFeatures() {
  const uploadArea = document.getElementById('uploadArea');
  const fileInput = document.getElementById('fileInput');
  const fileInfo = document.getElementById('fileInfo');
  const fileName = document.getElementById('fileName');
  const gradeCount = document.getElementById('gradeCount');
  const fillBtn = document.getElementById('fillBtn');
  const checkBtn = document.getElementById('checkBtn');
  const verifyBtn = document.getElementById('verifyBtn');
  const clearBtn = document.getElementById('clearBtn');
  const status = document.getElementById('status');

  // Enable check and verify buttons by default, clear when file uploaded
  checkBtn.disabled = false;
  verifyBtn.disabled = false;
  clearBtn.disabled = false;

  let gradesData = null;

  // File upload handlers
  uploadArea.addEventListener('click', () => fileInput.click());

  uploadArea.addEventListener('dragover', (e) => {
    e.preventDefault();
    uploadArea.classList.add('dragover');
  });

  uploadArea.addEventListener('dragleave', () => {
    uploadArea.classList.remove('dragover');
  });

  uploadArea.addEventListener('drop', (e) => {
    e.preventDefault();
    uploadArea.classList.remove('dragover');
    const file = e.dataTransfer.files[0];
    if (file) processFile(file);
  });

  fileInput.addEventListener('change', (e) => {
    const file = e.target.files[0];
    if (file) processFile(file);
  });

  function processFile(file) {
    const reader = new FileReader();
    reader.onload = function(e) {
      try {
        const data = new Uint8Array(e.target.result);
        const workbook = XLSX.read(data, { type: 'array' });

        // Get first sheet
        const firstSheetName = workbook.SheetNames[0];
        const worksheet = workbook.Sheets[firstSheetName];

        // 自动检测数据起始行：找到第一列（序号列）值为1的行
        const allData = XLSX.utils.sheet_to_json(worksheet, { header: 1 });
        let startRowIndex = 9; // 默认从第10行开始

        // 遍历找到序号为1的行
        for (let i = 0; i < allData.length; i++) {
          const row = allData[i];
          if (row && row.length > 0) {
            const serialNum = row[0];
            // 检查序号是否为1（可以是数字1或字符串"1"）
            if (serialNum === 1 || serialNum === '1') {
              startRowIndex = i;
              break;
            }
          }
        }

        console.log(`检测到数据起始行: 第${startRowIndex + 1}行`);

        // 从检测到的起始行开始解析
        const jsonData = XLSX.utils.sheet_to_json(worksheet, { header: 1, range: startRowIndex });

        gradesData = [];
        for (let i = 0; i < jsonData.length; i++) {
          const row = jsonData[i];
          if (row && row.length > 0) {
            // B列 (index 1) 是学号
            const studentNum = row[1];
            // C列 (index 2) 是姓名
            const studentName = row[2];
            // 最后一列是总成绩
            const totalScore = row[row.length - 1];

            // 验证学号是否存在
            if (studentNum !== undefined && studentNum !== null && studentNum !== '') {
              if (totalScore !== undefined && totalScore !== null && totalScore !== '') {
                gradesData.push({
                  index: i + 1,
                  studentNum: String(studentNum).trim(),
                  studentName: studentName ? String(studentName).trim() : '',
                  totalScore: parseFloat(totalScore)
                });
              }
            }
          }
        }

        // Update UI
        fileName.textContent = file.name;
        gradeCount.textContent = `已读取 ${gradesData.length} 条成绩记录`;
        fileInfo.style.display = 'block';
        fillBtn.disabled = false;

        showStatus(`成功读取 ${gradesData.length} 条成绩（从第${startRowIndex + 1}行开始）`, 'success');

        // Store in chrome storage
        chrome.storage.local.set({ gradesData: gradesData });
      } catch (error) {
        showStatus('解析Excel失败: ' + error.message, 'error');
      }
    };
    reader.readAsArrayBuffer(file);
  }

  // Fill grades button
  fillBtn.addEventListener('click', async () => {
    if (!gradesData || gradesData.length === 0) {
      showStatus('请先上传成绩文件', 'error');
      return;
    }

    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

      // Ensure content script is injected
      const ready = await ensureContentScript(tab.id);
      if (!ready) {
        showStatus('无法连接到页面，请确保在成绩录入页面', 'error');
        return;
      }

      // Send message to content script
      chrome.tabs.sendMessage(tab.id, {
        action: 'fillGrades',
        grades: gradesData
      }, (response) => {
        if (chrome.runtime.lastError) {
          showStatus('无法连接到页面，请刷新后重试', 'error');
        } else if (response && response.success) {
          showStatus(`😊 成功录入 ${response.count} 条成绩`, 'success');
        } else if (response && response.error) {
          showStatus(response.error, 'error');
        }
      });
    } catch (error) {
      showStatus('执行失败: ' + error.message, 'error');
    }
  });

  // Helper function to ensure content script is injected
async function ensureContentScript(tabId) {
  try {
    // Try to send a ping message first
    await chrome.tabs.sendMessage(tabId, { action: 'ping' });
    return true;
  } catch (e) {
    // Content script not responding, try to inject it
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tabId },
        files: ['content.js']
      });
      // Wait a bit for script to initialize
      await new Promise(resolve => setTimeout(resolve, 200));
      return true;
    } catch (e2) {
      return false;
    }
  }
}

// Check page status button
checkBtn.addEventListener('click', async () => {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    // Ensure content script is injected
    const ready = await ensureContentScript(tab.id);
    if (!ready) {
      showStatus('无法连接到页面，请确保在成绩录入页面', 'error');
      return;
    }

    chrome.tabs.sendMessage(tab.id, { action: 'checkPage' }, (response) => {
      if (chrome.runtime.lastError) {
        showStatus('无法连接到页面，请确保在成绩录入页面', 'error');
      } else if (response && response.hasInputFields) {
        showStatus(`😊 页面已就绪，检测到 ${response.inputCount} 个成绩输入框`, 'success');
      } else {
        showStatus('未检测到成绩输入框，请检查页面', 'info');
      }
    });
  } catch (error) {
    showStatus('检查失败: ' + error.message, 'error');
  }
});

  // Verify grades button
  verifyBtn.addEventListener('click', async () => {
    if (!gradesData || gradesData.length === 0) {
      showStatus('请先上传成绩文件', 'error');
      return;
    }

    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

      // Ensure content script is injected
      const ready = await ensureContentScript(tab.id);
      if (!ready) {
        showStatus('无法连接到页面，请确保在成绩录入页面', 'error');
        return;
      }

      chrome.tabs.sendMessage(tab.id, {
        action: 'getPageData'
      }, (response) => {
        if (chrome.runtime.lastError) {
          showStatus('无法连接到页面，请确保在成绩录入页面', 'error');
        } else if (response && response.success) {
          // Compare Excel data with page data
          const pageStudents = response.students;
          const mismatches = [];
          const conflictDetails = [];

          gradesData.forEach((excelStudent, index) => {
            const pageStudent = pageStudents[index];
            if (pageStudent) {
              const excelNum = String(excelStudent.studentNum).trim();
              const pageNum = String(pageStudent.studentNum).trim();
              const excelName = String(excelStudent.studentName || '').trim();
              const pageName = String(pageStudent.studentName || '').trim();
              const excelScore = excelStudent.totalScore;
              const pageScore = parseFloat(pageStudent.totalScore);

              let conflictType = [];
              if (excelNum !== pageNum) {
                conflictType.push('学号');
              }
              if (excelName !== pageName) {
                conflictType.push('姓名');
              }
              if (isNaN(pageScore) || Math.abs(excelScore - pageScore) > 0.01) {
                conflictType.push('成绩');
              }

              if (conflictType.length > 0) {
                mismatches.push(`第${index + 1}行：${conflictType.join('、')}不匹配`);
                conflictDetails.push({
                  row: index + 1,
                  excelStudentNum: excelNum,
                  excelStudentName: excelName,
                  excelScore: excelScore,
                  pageStudentNum: pageNum,
                  pageStudentName: pageName,
                  pageScore: isNaN(pageScore) ? '' : pageScore,
                  conflictType: conflictType.join('、')
                });
              }
            }
          });

          if (mismatches.length === 0) {
            showStatus('😊 学号、姓名、成绩已经全部核实', 'success');
          } else {
            const summary = `⚠ 共 ${gradesData.length} 条记录，${mismatches.length} 条不匹配`;
            showStatus(summary, 'error');

            // Create conflict Excel file
            const ws = XLSX.utils.aoa_to_sheet([
              ['行号', 'Excel学号', 'Excel姓名', 'Excel成绩', '页面学号', '页面姓名', '页面成绩', '不匹配项目']
            ]);
            conflictDetails.forEach(item => {
              XLSX.utils.sheet_add_aoa(ws, [[
                item.row,
                item.excelStudentNum,
                item.excelStudentName,
                item.excelScore,
                item.pageStudentNum,
                item.pageStudentName,
                item.pageScore,
                item.conflictType
              ]], { skipHeader: true, origin: -1 });
            });

            const wb = XLSX.utils.book_new();
            XLSX.utils.book_append_sheet(wb, ws, '数据冲突');

            // Generate filename with timestamp
            const timestamp = new Date().toISOString().slice(0, 19).replace(/[:-]/g, '');
            XLSX.writeFile(wb, `数据冲突检查_${timestamp}.xlsx`);

            console.log('数据不匹配详情：', mismatches);
          }
        } else if (response && response.error) {
          showStatus(response.error, 'error');
        }
      });
    } catch (error) {
      showStatus('检查失败: ' + error.message, 'error');
    }
  });

  // Clear grades button
  clearBtn.addEventListener('click', async () => {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

      // Ensure content script is injected
      const ready = await ensureContentScript(tab.id);
      if (!ready) {
        showStatus('无法连接到页面，请确保在成绩录入页面', 'error');
        return;
      }

      chrome.tabs.sendMessage(tab.id, { action: 'clearGrades' }, (response) => {
        if (chrome.runtime.lastError) {
          showStatus('无法连接到页面，请确保在成绩录入页面', 'error');
        } else if (response && response.success) {
          showStatus(`😊 成功清除 ${response.count} 个成绩`, 'success');
        } else if (response && response.error) {
          showStatus(response.error, 'error');
        }
      });
    } catch (error) {
      showStatus('清除失败: ' + error.message, 'error');
    }
  });

  function showStatus(message, type) {
    status.textContent = message;
    status.className = 'status ' + type;
  }
}
