# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with this repository.

## Project Overview

Chrome extension for automatic grade input from Excel files into university portal grade entry pages.

## Commands

```bash
# Install extension in Chrome
# 1. Open chrome://extensions/
# 2. Enable Developer Mode
# 3. Click "Load unpacked"
# 4. Select grade-input-extension folder

# Git operations
git add -A && git commit -m "message"  # Commit changes
git tag v0.0.0                         # Create release tag
```

## Generate Auth Code for Distribution

```bash
# Generate 14-day auth code (default)
python3 generate_auth_code.py

# Generate with custom days
python3 generate_auth_code.py 30
```

## Architecture

This is a **Chrome Extension (Manifest V3)** with the following components:

### Extension Components
- **manifest.json** - Extension configuration, permissions, content script registration
- **popup.html/popup.js** - Extension popup UI; handles Excel file upload via FileReader + SheetJS; authorization verification
- **content.js** - Injected into all pages; finds grade input fields via CSS selectors, fills/clears values
- **background.js** - Service worker (minimal, mainly for extension lifecycle)

### Authorization System

**popup.js** includes an authorization verification module:
- Auth code format: `Base64(XOR-encrypted(JSON({expireDate, salt})))`
- First-time users must enter auth code to unlock features
- Auth code validity: 14 days (default)
- Stored in `chrome.storage.local`

### Data Flow
1. User uploads Excel in popup → `popup.js` parses with SheetJS
2. **Mode Detection**: Automatically detects single or dual column mode
   - Single column: Reads last column as score
   - Dual column: Reads 5th-from-end column as score1, 3rd-from-end as score2
3. Popup sends message to content script via `chrome.tabs.sendMessage`
4. Content script detects page mode and fills values accordingly

### Key Implementation Details

**Excel Parsing** (`popup.js`):
- Uses `XLSX.read()` with `type: 'array'`
- Reads first sheet, skips first 9 rows (`range: 9`)
- **Mode Detection**: `detectExcelMode()` checks if 5th-from-end and 3rd-from-end columns have valid data
- **Single Column Mode**: Extracts last column as total score
- **Dual Column Mode**: Extracts 5th-from-end as `score1`, 3rd-from-end as `score2`
- Uses `chrome.storage.local` to store `gradesData` and `gradesMode`

**Authorization** (`popup.js`):
- `generateAuthCode()` - Generates auth codes (used by `generate_auth_code.py`)
- `verifyAuthCode()` - Validates auth codes
- `checkAuth()` - Checks authorization status from storage

**Input Field Detection** (`content.js`):
- Tries multiple CSS selectors: `input[name*="GradeState"]`, `input.txtScore`, `input[name*="examGrade-"][name$=".score"]`, etc.
- Filters by visibility and enabled state
- `getPageData()` uses multiple patterns to extract student info and scores:
  - Pattern 1: Struts2 indexed properties (`examGradeState[0].student.code`)
  - Pattern 2: Table rows - extracts student number/name from cells, scores from `<input>` elements
  - Pattern 3: Table rows with examGrade-XXX.score format
  - Fallback: Groups inputs by DOM position

**Grade Filling** (`content.js`):
- `detectPageMode()` - Automatically detects single or dual column mode by:
  - Extracting student indices from input name attributes
  - Calculating input-to-student ratio (>=1.5 indicates dual column)
- `organizeFieldsByRow()` - Groups inputs by DOM position (using `getBoundingClientRect.top`)
- `fillSingleMode()` - Sequential filling with 100ms delay
- `fillDualMode()` - Fills inputs row by row, two fields per student
- Dispatches `change` and `input` events after setting values

**Verification** (`popup.js`):
- `verifyGrades()` supports both single and dual column modes
- Dual column verification compares both `score1` and `score2`
- Conflict export includes score1/score2 columns for dual column mode

## Related Projects

The target grade entry page (`grade-input.html`) uses:
- jQuery, jQuery UI, Backbone.js, Underscore.js
- Struts2 backend (`/eams/teach/grade/course/teacher!inputGA.action`)
