// Content script for grade input extension

let inputFields = [];
let currentIndex = 0;

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'ping') {
    sendResponse({ pong: true });
    return false;
  } else if (request.action === 'fillGrades') {
    fillGrades(request.grades, sendResponse);
    return true; // Will respond asynchronously
  } else if (request.action === 'checkPage') {
    checkPage(sendResponse);
    return false;
  } else if (request.action === 'clearGrades') {
    clearGrades(sendResponse);
    return true; // Will respond asynchronously
  } else if (request.action === 'getPageData') {
    getPageData(sendResponse);
    return true; // Will respond asynchronously
  }
});

// ==================== 页面模式检测 ====================

/**
 * 检测页面是单列还是双列成绩输入模式
 * @returns {Object} { mode: 'single' | 'dual', inputCount: number, studentCount: number }
 */
function detectPageMode() {
  const fields = findInputFields();
  if (fields.length === 0) {
    return { mode: 'unknown', inputCount: 0, studentCount: 0 };
  }

  // 策略1: 检查输入框 name 属性中的索引，统计唯一学生数
  const studentIndices = new Set();
  const indexedPattern = /examGradeState\[(\d+)\]/;

  fields.forEach(field => {
    const match = field.name.match(indexedPattern);
    if (match) {
      studentIndices.add(parseInt(match[1]));
    }
  });

  const detectedStudentCount = studentIndices.size;

  // 策略2: 检查输入框数量与学生数的比例
  // 双列模式：输入框数量 >= 学生数 * 2
  // 单列模式：输入框数量 ≈ 学生数
  if (detectedStudentCount > 0) {
    const ratio = fields.length / detectedStudentCount;
    if (ratio >= 1.5) {
      return { mode: 'dual', inputCount: fields.length, studentCount: detectedStudentCount };
    } else {
      return { mode: 'single', inputCount: fields.length, studentCount: detectedStudentCount };
    }
  }

  // 策略3: 如果无法从 name 属性提取学生数，按 DOM 位置分组后判断
  const rows = organizeFieldsByRow(fields);
  if (rows.length > 0) {
    const inputsPerRow = rows[0].length;
    if (inputsPerRow >= 2) {
      return { mode: 'dual', inputCount: fields.length, studentCount: rows.length };
    }
  }

  // 默认单列模式
  return { mode: 'single', inputCount: fields.length, studentCount: Math.ceil(fields.length / 1.5) };
}

/**
 * 按行组织输入框（通过元素的 getBoundingClientRect.top 判断）
 * @param {Array} fields - 输入框数组
 * @returns {Array} 按行分组的输入框数组
 */
function organizeFieldsByRow(fields) {
  const rows = [];
  const rowMap = new Map();

  fields.forEach((field) => {
    const element = field.element;
    const rect = element.getBoundingClientRect();
    const rowKey = Math.round(rect.top);

    if (!rowMap.has(rowKey)) {
      rowMap.set(rowKey, []);
    }
    rowMap.get(rowKey).push(field);
  });

  // 按顶部位置排序
  const sortedKeys = Array.from(rowMap.keys()).sort((a, b) => a - b);

  sortedKeys.forEach(key => {
    const rowFields = rowMap.get(key);
    rows.push(rowFields);
  });

  return rows;
}

function findInputFields() {
  inputFields = [];

  // Look for various types of grade input fields
  const possibleSelectors = [
    // Direct input fields
    'input[type="text"][name*="Grade"]',
    'input[type="text"][name*="grade"]',
    'input[type="number"][name*="Grade"]',
    'input[type="number"][name*="grade"]',
    'input[type="text"][class*="grade"]',
    'input[type="number"][class*="grade"]',

    // Struts2 form fields (common in academic systems)
    'input[name*="examGradeState"]',
    'input[name*="GradeState"]',

    // Editable table cells
    'td[contenteditable="true"]',
    'div[contenteditable="true"]',

    // Common grade input patterns
    'input.txtScore',
    'input.score-input',
    'input.grade-input',

    // All text/number inputs in forms
    'form input[type="text"]',
    'form input[type="number"]'
  ];

  // Try each selector
  for (const selector of possibleSelectors) {
    const fields = document.querySelectorAll(selector);
    fields.forEach(field => {
      // Check if it's visible and enabled
      if (isVisible(field) && !field.disabled && !field.readOnly) {
        // Avoid duplicates
        const existing = inputFields.find(f => f.name === field.name && f.value === field.value);
        if (!existing) {
          inputFields.push({
            element: field,
            name: field.name || field.className,
            value: field.value
          });
        }
      }
    });
  }

  // Also try to find by table structure
  const tables = document.querySelectorAll('table');
  tables.forEach(table => {
    const rows = table.querySelectorAll('tr');
    rows.forEach(row => {
      const cells = row.querySelectorAll('td');
      cells.forEach((cell, index) => {
        // Check if cell looks like a grade cell
        const input = cell.querySelector('input[type="text"], input[type="number"]');
        if (input && isVisible(input)) {
          // Only add if not already found
          const exists = inputFields.find(f => f.element === input);
          if (!exists && !input.disabled && !input.readOnly) {
            inputFields.push({
              element: input,
              name: input.name || `cell_${index}`,
              value: input.value,
              parentCell: cell
            });
          }
        }
      });
    });
  });

  return inputFields;
}

function isVisible(element) {
  const style = window.getComputedStyle(element);
  return style.display !== 'none' &&
         style.visibility !== 'hidden' &&
         style.opacity !== '0' &&
         element.offsetParent !== null;
}

function fillGrades(grades, sendResponse) {
  try {
    // 检测页面模式
    const pageMode = detectPageMode();
    console.log('检测到页面模式:', pageMode);

    // Find all input fields
    const fields = findInputFields();

    if (fields.length === 0) {
      sendResponse({
        success: false,
        error: '未找到成绩输入框，请确保页面已加载完成'
      });
      return;
    }

    let filledCount = 0;
    currentIndex = 0;

    if (pageMode.mode === 'dual') {
      // 双列模式：按行组织输入框，每行两个成绩输入框
      filledCount = fillDualMode(grades, fields);
    } else {
      // 单列模式：顺序填充
      filledCount = fillSingleMode(grades, fields);
    }

    // 延迟返回结果，确保填充完成
    setTimeout(() => {
      sendResponse({
        success: true,
        count: filledCount,
        totalGrades: grades.length,
        totalFields: fields.length,
        pageMode: pageMode.mode
      });
    }, grades.length * 110);

  } catch (error) {
    sendResponse({
      success: false,
      error: '录入失败: ' + error.message
    });
  }
}

/**
 * 单列模式填充
 * @param {Array} grades - 成绩数据数组
 * @param {Array} fields - 输入框数组
 * @returns {number} 成绩记录数（不是输入框数量）
 */
function fillSingleMode(grades, fields) {
  currentIndex = 0;

  function fillNext() {
    if (currentIndex < grades.length && currentIndex < fields.length) {
      const grade = grades[currentIndex];
      const field = fields[currentIndex];

      // Fill the input
      field.element.value = grade.totalScore;

      // Trigger change events
      dispatchEvents(field.element);

      currentIndex++;

      // Delay before next fill
      setTimeout(fillNext, 100);
    }
  }

  fillNext();
  return grades.length;  // 返回成绩记录数
}

/**
 * 双列模式填充
 * @param {Array} grades - 成绩数据数组
 * @param {Array} fields - 输入框数组
 * @returns {number} 成绩记录数（不是输入框数量）
 */
function fillDualMode(grades, fields) {
  // 按行组织输入框
  const rows = organizeFieldsByRow(fields);
  currentIndex = 0;

  function fillNext() {
    if (currentIndex < grades.length && currentIndex < rows.length) {
      const grade = grades[currentIndex];
      const rowFields = rows[currentIndex];

      // 填充第一列成绩（平时成绩）到第一个输入框
      if (rowFields[0] && grade.score1 !== undefined) {
        rowFields[0].element.value = grade.score1;
        dispatchEvents(rowFields[0].element);
      }

      // 填充第二列成绩（期末成绩）到第二个输入框
      if (rowFields[1] && grade.score2 !== undefined) {
        rowFields[1].element.value = grade.score2;
        dispatchEvents(rowFields[1].element);
      }

      currentIndex++;

      // Delay before next fill
      setTimeout(fillNext, 100);
    }
  }

  fillNext();
  return grades.length;  // 返回成绩记录数
}

/**
 * 触发输入和变更事件
 */
function dispatchEvents(element) {
  const changeEvent = new Event('change', { bubbles: true });
  element.dispatchEvent(changeEvent);

  const inputEvent = new Event('input', { bubbles: true });
  element.dispatchEvent(inputEvent);
}

function checkPage(sendResponse) {
  const fields = findInputFields();
  sendResponse({
    hasInputFields: fields.length > 0,
    inputCount: fields.length,
    fieldNames: fields.slice(0, 10).map(f => f.name) // Send first 10 names
  });
}

function clearGrades(sendResponse) {
  try {
    // Find all input fields
    const fields = findInputFields();

    if (fields.length === 0) {
      sendResponse({
        success: false,
        error: '未找到成绩输入框'
      });
      return;
    }

    let clearedCount = 0;
    currentIndex = 0;

    // Clear grades sequentially with delay
    function clearNext() {
      if (currentIndex < fields.length) {
        const field = fields[currentIndex];

        // Only clear if field has a value
        if (field.element.value && field.element.value !== '') {
          field.element.value = '';

          // Trigger change events
          const event = new Event('change', { bubbles: true });
          field.element.dispatchEvent(event);

          const inputEvent = new Event('input', { bubbles: true });
          field.element.dispatchEvent(inputEvent);

          clearedCount++;
        }

        currentIndex++;

        // Delay before next clear
        setTimeout(clearNext, 50);
      } else {
        sendResponse({
          success: true,
          count: clearedCount
        });
      }
    }

    clearNext();

  } catch (error) {
    sendResponse({
      success: false,
      error: '清除失败: ' + error.message
    });
  }
}

function getPageData(sendResponse) {
  try {
    const students = [];

    // Try to find student data using various patterns
    // Pattern 1: Struts2 indexed properties like examGradeState[0].student.code
    const indexedPattern = /examGradeState\[(\d+)\]\.(\w+)/;
    const allInputs = document.querySelectorAll('input[name]');

    // Collect all indexed fields
    const indexedFields = {};
    allInputs.forEach(input => {
      const match = input.name.match(indexedPattern);
      if (match) {
        const index = parseInt(match[1]);
        const fieldType = match[2]; // e.g., 'student.code', 'examGrade.percent'
        if (!indexedFields[index]) {
          indexedFields[index] = {};
        }
        indexedFields[index][fieldType] = input.value;
      }
    });

    // Convert to student array
    Object.keys(indexedFields).sort((a, b) => parseInt(a) - parseInt(b)).forEach(index => {
      const fields = indexedFields[index];
      const studentNum = fields['student.code'] || fields['student.std.code'] || fields['studentNo'] || '';
      const studentName = fields['student.name'] || fields['student.std.name'] || fields['studentName'] || '';
      const totalScore = fields['examGrade.percent'] || fields['score'] || fields['grade'] || '';
      // 双列模式：尝试获取两个成绩字段
      const score1 = fields['examGrade.percent'] || fields['score1'] || fields['grade1'] || '';
      const score2 = fields['examGrade.percent2'] || fields['score2'] || fields['grade2'] || '';

      students.push({
        index: parseInt(index) + 1,
        studentNum: studentNum,
        studentName: studentName,
        totalScore: totalScore,
        score1: score1,
        score2: score2
      });
    });

    // Pattern 2: Table-based with input field values
    // 成绩在 input 元素中，不在 td 单元格文本中
    if (students.length === 0) {
      const tables = document.querySelectorAll('table');
      tables.forEach(table => {
        const rows = table.querySelectorAll('tr');
        rows.forEach((row, rowIndex) => {
          const cells = row.querySelectorAll('td, th');

          if (cells.length >= 2) {
            // 从单元格获取学号和姓名
            const studentNumCell = cells[0].textContent.trim();
            const studentNameCell = cells[1].textContent.trim();

            // 检查是否像学号（纯数字）
            if (/^\d+$/.test(studentNumCell) && studentNumCell.length > 0) {
              // 在当前行查找成绩输入框（支持 examGrade-XXX.score 格式）
              const inputs = row.querySelectorAll('input[name*="examGrade-"][name$=".score"]');

              // 双列模式：两个输入框；单列模式：一个输入框
              let score1 = '', score2 = '', totalScore = '';

              if (inputs.length >= 2) {
                score1 = inputs[0].value;
                score2 = inputs[1].value;
                totalScore = score1 || score2;
              } else if (inputs.length === 1) {
                totalScore = inputs[0].value;
              }

              students.push({
                index: rowIndex + 1,
                studentNum: studentNumCell,
                studentName: studentNameCell,
                totalScore: totalScore,
                score1: score1,
                score2: score2
              });
            }
          }
        });
      });
    }

    // Pattern 3: Find by table row structure with input fields (备用方案)
    if (students.length === 0) {
      const tables = document.querySelectorAll('table');
      tables.forEach(table => {
        const rows = table.querySelectorAll('tr');
        rows.forEach((row, rowIndex) => {
          // 查找成绩输入框
          const inputs = row.querySelectorAll('input[name*="examGrade-"][name$=".score"]');
          if (inputs.length > 0) {
            let score1 = '', score2 = '';

            if (inputs.length >= 2) {
              score1 = inputs[0].value;
              score2 = inputs[1].value;
            } else if (inputs.length === 1) {
              score1 = inputs[0].value;
            }

            students.push({
              index: rowIndex + 1,
              studentNum: '',
              studentName: '',
              totalScore: score1 || score2 || '',
              score1: score1,
              score2: score2
            });
          }
        });
      });
    }

    // Fallback: 按 DOM 位置组织输入框
    if (students.length === 0) {
      const fields = findInputFields();
      const rows = organizeFieldsByRow(fields);
      rows.forEach((rowFields, idx) => {
        students.push({
          index: idx + 1,
          studentNum: '',
          studentName: '',
          totalScore: rowFields[0]?.element?.value || '',
          score1: rowFields[0]?.element?.value || '',
          score2: rowFields[1]?.element?.value || ''
        });
      });
    }

    sendResponse({
      success: true,
      students: students,
      count: students.length
    });

  } catch (error) {
    sendResponse({
      success: false,
      error: '获取页面数据失败: ' + error.message
    });
  }
}

// Also try to find fields on page load
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    setTimeout(() => findInputFields(), 500);
  });
} else {
  setTimeout(() => findInputFields(), 500);
}
